<?php
/*
Plugin Name: Formverse Connect
Plugin URI: https://formverse.gr/integrations/wordpress
Description: Connect your WordPress site to Formverse. List your forms, embed with a shortcode, view analytics, and more.
Version: 1.2
Author: Formverse
Author URI: https://formverse.gr/
Text Domain: formverse-connect
Domain Path: /languages
Requires at least: 5.4
Tested up to: 6.8
Requires PHP: 7.4
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Tags: forms, embed, form builder, analytics, formverse
*/

if (!defined('ABSPATH')) exit;

// ---- Settings Page ----

add_action('admin_menu', function() {
	add_options_page('Formverse Connect', 'Formverse', 'manage_options', 'formverse-connect', 'formverse_settings_page');
});

add_action('enqueue_block_editor_assets', function() {
	wp_enqueue_script(
		'formverse-block',
		plugins_url('formverse-block.js', __FILE__),
		array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n'),
		false,
		true
	);
});
add_action('rest_api_init', function() {
	register_rest_route('formverse/v1', '/forms', array(
		'methods' => 'GET',
		'permission_callback' => function() { return current_user_can('edit_posts'); },
		'callback' => function() {
			$forms = formverse_get_forms();
			return rest_ensure_response(array('forms' => $forms));
		}
	));
});
// Add TinyMCE button
add_action('admin_init', function() {
	add_filter('mce_external_plugins', function($plugins) {
		$plugins['formverse'] = plugins_url('formverse-tinymce.js', __FILE__);
		return $plugins;
	});
	add_filter('mce_buttons', function($buttons) {
		array_push($buttons, 'formverse');
		return $buttons;
	});
});
add_action('wp_ajax_formverse_get_forms_for_editor', function() {
	if (!current_user_can('edit_posts')) { wp_send_json(['forms'=>[]]); }
	wp_send_json(['forms' => formverse_get_forms()]);
});


function formverse_settings_page() {
	?>
	<div class="wrap">
		<h1>Formverse Connect</h1>
		<form method="post" action="options.php">
			<?php settings_fields('formverse_settings'); ?>
			<?php do_settings_sections('formverse_settings'); ?>
			<table class="form-table">
				<tr valign="top">
					<th scope="row">API Key</th>
					<td>
						<input type="text" name="formverse_api_key" value="<?php echo esc_attr(get_option('formverse_api_key')); ?>" size="60"/>
						<p class="description">You can generate your API key in your Formverse dashboard at <a href="https://formverse.gr/" target="_blank">formverse.gr</a></p>
					</td>
				</tr>
			</table>
			<?php submit_button(); ?>
		</form>
		<hr>
		<h2>Your Forms</h2>
		<?php formverse_display_forms_list(); ?>
	</div>
	<?php
}

add_action('admin_init', function() {
	register_setting('formverse_settings', 'formverse_api_key');
});

// ---- Fetch Forms Helper ----

function formverse_get_forms() {
	$api_key = get_option('formverse_api_key');
	$api_url = 'https://formverse.gr/api';
	if (!$api_key || !$api_url) return [];

	$response = wp_remote_get("$api_url/forms", [
		'headers' => [
			'X-API-KEY' => $api_key
		],
		'timeout' => 15
	]);
	if (is_wp_error($response)) return [];
	$body = wp_remote_retrieve_body($response);
	$data = json_decode($body, true);
	if (!isset($data['forms'])) return [];
	return $data['forms'];
}

function formverse_get_form_stats($id) {
	$api_key = get_option('formverse_api_key');
	$api_url = 'https://formverse.gr/api';

	$response = wp_remote_get("$api_url/forms/$id/stats", [
		'headers' => [
			'X-API-KEY' => $api_key
		],
		'timeout' => 15
	]);
	if (is_wp_error($response)) return [];
	$body = wp_remote_retrieve_body($response);
	$data = json_decode($body, true);
	return $data;
}

// ---- Forms List (Admin) ----

function formverse_display_forms_list() {
	$forms = formverse_get_forms();
	if (!$forms) {
		echo '<p>No forms found or could not connect to API.</p>';
		return;
	}
	// Modal for Analytics (displayed once)
	?>
	<div id="formverse-analytics-modal" style="display:none;position:fixed;top:0;left:0;width:100vw;height:100vh;background:rgba(0,0,0,0.4);z-index:9999;align-items:center;justify-content:center;">
		<div id="formverse-analytics-content" style="background:#fff;max-width:650px;margin:auto;padding:2em;position:relative;">
			<button id="formverse-analytics-close" style="position:absolute;top:8px;right:8px;font-size:2em;line-height:1;">&times;</button>
			<div id="formverse-analytics-inner"></div>
		</div>
	</div>
	<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
	<script>
		document.addEventListener('DOMContentLoaded', function() {
			var modal = document.getElementById('formverse-analytics-modal');
			var closeBtn = document.getElementById('formverse-analytics-close');
			closeBtn.onclick = function() {
				modal.style.display = 'none';
			}
			window.formverseShowAnalytics = function(formId, formTitle) {
				var inner = document.getElementById('formverse-analytics-inner');
				inner.innerHTML = '<div style="text-align:center"><em>Loading analytics...</em></div>';
				modal.style.display = 'flex';

				// AJAX
				var xhr = new XMLHttpRequest();
				xhr.open('GET', '<?php echo admin_url('admin-ajax.php'); ?>?action=formverse_get_analytics&form_id=' + encodeURIComponent(formId));
				xhr.onload = function() {
					if(xhr.status === 200) {
						var data = JSON.parse(xhr.responseText);
						inner.innerHTML = formverseFormatAnalytics(data, formTitle);
						formverseRenderChart(data);
					} else {
						inner.innerHTML = '<div style="color:#a00;">Could not load analytics.</div>';
					}
				};
				xhr.send();
			}

			window.formverseFormatAnalytics = function(data, formTitle) {
				if(!data || typeof data !== 'object' || data.error){
					return '<div style="color:#a00;">No analytics data available.</div>';
				}
				let html = '';
				html += '<h3>Analytics for: ' + formTitle + '</h3>';
				html += '<div style="display: flex; flex-wrap: wrap; gap: 1em; margin-bottom: 2em;">';

				// Info boxes for basic stats
				const statLabels = {
					views: 'Views',
					submits: 'Submissions',
					abandons: 'Abandons',
					averageTime: 'Avg. Time (sec)'
				};
				for(const key of ['views','submits','abandons','averageTime']) {
					if(typeof data[key] !== 'undefined') {
						html += `<div style="flex:1 1 150px; background:#f9f9f9; border:1px solid #eee; border-radius:10px; padding:1em; min-width:130px; text-align:center;">
                        <div style="font-size:1.6em;font-weight:bold;">${data[key]}</div>
                        <div style="color:#888">${statLabels[key]||key}</div>
                    </div>`;
					}
				}
				html += '</div>';

				// Mini-tables for countries, browsers, referrers
				if(data.countries && typeof data.countries === 'object') {
					html += '<h4>Countries</h4><table style="width:100%;margin-bottom:1em;"><thead><tr><th style="text-align:left;">Country</th><th style="text-align:right;">Count</th></tr></thead><tbody>';
					for(let k in data.countries) {
						html += `<tr><td>${k === "" ? "Unknown" : k}</td><td style="text-align:right;">${data.countries[k]}</td></tr>`;
					}
					html += '</tbody></table>';
				}
				if(data.browsers && typeof data.browsers === 'object') {
					html += '<h4>Browsers</h4><table style="width:100%;margin-bottom:1em;"><thead><tr><th style="text-align:left;">Browser</th><th style="text-align:right;">Count</th></tr></thead><tbody>';
					for(let k in data.browsers) {
						html += `<tr><td>${k}</td><td style="text-align:right;">${data.browsers[k]}</td></tr>`;
					}
					html += '</tbody></table>';
				}
				if(data.referrers && typeof data.referrers === 'object') {
					html += '<h4>Referrers</h4><table style="width:100%;margin-bottom:1em;"><thead><tr><th style="text-align:left;">Referrer</th><th style="text-align:right;">Count</th></tr></thead><tbody>';
					for(let k in data.referrers) {
						html += `<tr><td style="max-width:250px;overflow-x:auto;white-space:nowrap;">${k}</td><td style="text-align:right;">${data.referrers[k]}</td></tr>`;
					}
					html += '</tbody></table>';
				}

				// Trends (graph placeholder)
				if(data.trends && typeof data.trends === 'object') {
					html += '<h4>Daily Trends</h4>';
					html += '<canvas id="formverse-analytics-chart" style="max-width:100%;min-height:200px;max-height:300px;"></canvas>';
				}

				return html;
			}

			window.formverseRenderChart = function(data) {
				if(!data.trends || typeof data.trends !== 'object') return;
				const ctx = document.getElementById('formverse-analytics-chart');
				if(!ctx) return;

				// Sort dates
				const labels = Object.keys(data.trends).sort();
				const values = labels.map(d => data.trends[d]);

				if(window.formverseChart && typeof window.formverseChart.destroy === 'function') {
					window.formverseChart.destroy();
				}

				window.formverseChart = new Chart(ctx, {
					type: 'line',
					data: {
						labels: labels,
						datasets: [{
							label: 'Submissions/Day',
							data: values,
							fill: true,
							tension: 0.25
						}]
					},
					options: {
						responsive: true,
						plugins: {legend: {display: false}},
						scales: {
							x: { title: { display: false }},
							y: { beginAtZero: true }
						}
					}
				});
			}
		});
	</script>
	<?php
	// Table
	echo '<table class="widefat fixed" style="max-width:100%;">';
	echo '<thead><tr>
        <th>ID</th>
        <th>Admin Title</th>
        <th>Submissions</th>
        <th>Embed URL</th>
        <th>Embed Shortcode</th>
        <th>Analytics</th>
    </tr></thead>';
	echo '<tbody>';
	foreach ($forms as $form) {
		$form_id = esc_html($form['id']);
		$admin_title = "<a href='https://formverse.gr/app/forms/{$form['id']}/edit'>".esc_html($form['admin_title'] ?? $form['name'] ?? $form['title'] ?? 'Untitled')."</a>";
		$submissions = isset($form['submissions']) ? intval($form['submissions']) : 0;
		$embed_url = esc_html($form['embed_url'] ?? '');
		$form_title = esc_attr($admin_title);

		echo '<tr>';
		echo "<td>$form_id</td>";
		echo "<td>$admin_title</td>";
		echo "<td>$submissions</td>";
		echo "<td><input type='text' readonly value='$embed_url' style='width:250px;'></td>";
		echo "<td><code>[formverse_form id=\"$form_id\"]</code></td>";
		echo "<td><button type='button' class='button' onclick='formverseShowAnalytics(\"$form_id\", \"$form_title\")'>Show Analytics</button></td>";
		echo '</tr>';
	}
	echo '</tbody></table>';
	echo '<p>Use the <code>[formverse_form id="..."]</code> shortcode to embed a form into your posts or pages.</p>';
}

// ---- AJAX for Analytics ----

add_action('wp_ajax_formverse_get_analytics', function() {
	if (!current_user_can('manage_options')) {
		wp_send_json_error(['error' => 'Unauthorized'], 403);
	}
	$form_id = isset($_GET['form_id']) ? sanitize_text_field($_GET['form_id']) : '';
	if (!$form_id) {
		wp_send_json_error(['error' => 'No form ID'], 400);
	}
	$stats = formverse_get_form_stats($form_id);
	wp_send_json($stats);
});

// ---- Shortcode ----

add_shortcode('formverse_form', function($atts) {
	$atts = shortcode_atts(['id' => ''], $atts);
	$form_id = sanitize_text_field($atts['id']);
	if (!$form_id) return '';

	$forms = formverse_get_forms();
	foreach ($forms as $form) {
		if ((string)$form['id'] === $form_id) {
			$embed_url = $form['embed_url'] ?? '';
			if ($embed_url) {
				return '<iframe src="' . esc_url($embed_url) . '" width="100%" height="650" frameborder="0" style="border:0;background:transparent;min-width:320px;min-height:300px;overflow:hidden;" allowtransparency="true" loading="lazy" title="Formverse Form"></iframe>';
			}
		}
	}
	return '<em>Form not found.</em>';
});